/*
 * @brief Secondary loader protocol structures and manifest constants
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */
#ifndef _SL_PROTOCOL_H_
#define _SL_PROTOCOL_H_

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/** Structure used to setup host interface for the secondary loader. The SPI
   port/pin selections for SSEL, SCK, MOSI, and MISO only need to be configured
   if ifSel is SL_SPI0 or SL_SPI1. */
typedef struct {
	uint8_t img_type;		/*!< Image type (SL_IMAGE_T)  */
	uint8_t ifSel;			/*!< Interface selection for host (SL_IFSEL_T) */
	uint8_t hostIrqPortPin;	/*!< Host IRQ port (bits 7:5) and pins (bits 4:0) */
	uint8_t hostMisoPortPin;	/*!< SPI MISO port (bits 7:5) and pins (bits 4:0) */
	uint8_t hostMosiPortPin;	/*!< SPI MOSI port (bits 7:5) and pins (bits 4:0) */
	uint8_t hostSselPortPin;	/*!< SPI SEL port (bits 7:5) and pins (bits 4:0) */
	uint8_t hostSckPortPin;		/*!< SPI SCK port (bits 7:5) and pins (bits 4:0) */
	uint8_t checksum;		/*!< Checksum. XOR of the remaining 7 bytes of this structure. */
} SL_PINSETUP_T;



typedef bool (*InBootSecondaryLoader)(const SL_PINSETUP_T *pSetup);

/* Address of indtrect boot table */
#define SL_INDIRECT_FUNC_TABLE      (0x00001F00)

/* Placement addresses for app call flag and app supplied config daa
   for host interface pins. Note these addresses may be used in the
   startup code source and may need values changed there also. */
#define SL_ADDRESS_APPCALLEDFL      (0x10000000)
#define SL_ADDRESS_APPPINDATA       (0x10000004)

/* Function for booting the secondary loader from an application. Returns with
   false if the pSetup strructure is not valid, or doesn't return if the
   loader was started successfully. */
STATIC INLINE bool bootSecondaryLoader(const SL_PINSETUP_T *pSetup)
{
	IRQn_Type irqNum;
	InBootSecondaryLoader SL, *pSL = (InBootSecondaryLoader *) SL_INDIRECT_FUNC_TABLE;
	SL_PINSETUP_T *pAppPinSetup = (SL_PINSETUP_T *) SL_ADDRESS_APPPINDATA;

	*pAppPinSetup = *pSetup;

	SL = *pSL;
	/* Disable sysTick */
	SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;

	/* Disable all NVIC interrupts in case this loader is called from the
	   user application */
	for (irqNum = SPI0_IRQn; irqNum <= PIN_INT7_IRQn; irqNum++) {
		NVIC_DisableIRQ(irqNum);
	}
	return SL(pSetup);
}

/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif  /* _SL_PROTOCOL_H_ */
